﻿//Copyright (C) 2010  Jonathan Preece
//
//This program is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program.  If not, see <http://www.gnu.org/licenses/>.

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using Timer = System.Windows.Forms.Timer;

namespace RevisionAnalyser.Global.EasterEgg
{
    public class SnakeGame
    {
        private readonly PictureBox _canvas;

        private readonly Brush _bodyBrush = new SolidBrush(Color.LimeGreen);
        private readonly Brush _headBrush = new SolidBrush(Color.Red);
        private readonly Brush _foodBrush = new SolidBrush(Color.Yellow);
        private readonly Brush _scoreBrush = new SolidBrush(Color.Gray);

        private readonly Random _random = new Random(Environment.TickCount);

        private readonly Snake _snake = new Snake();
        private int _score;

        private readonly Timer _timer = new Timer();
        private Direction _currentDirection = Direction.Right;

        private Segment _foodSegment;

        private List<Segment> _grid;
        private bool _isFoodSegmentVisible;

        #region Constructor

        public SnakeGame(PictureBox canvas)
        {
            #region Init Controls

            _canvas = canvas;
            _canvas.Paint += Canvas_Paint;

            _timer = new Timer();
            _timer.Tick += Timer_Tick;
            _timer.Interval = 3000;
            _timer.Enabled = true;
            _timer.Start();

            InitializeGrid();

            #endregion

            _snake.Reset();
            _snake.MovingDirection = _currentDirection;
        }

        #endregion

        public void SetDirection(Direction newDirection)
        {
            if (IsCompatible(newDirection))
            {
                _currentDirection = newDirection;
                _snake.MovingDirection = newDirection;
            }
        }

        private void Canvas_Paint(object sender, PaintEventArgs e)
        {
            //Check for collisions before painting
            //This ensures the snake actually looks like its hit the side before it happens

            if (IsGameOver())
            {
                e.Graphics.DrawString("Game Over", new Font(FontFamily.GenericSansSerif, 24, FontStyle.Bold), _scoreBrush, 190, 150);
                return;
            }

            //Advance the position of the snake, using the given direction
            MoveSnake();

            //Draw the food, if visible
            if (_isFoodSegmentVisible)
                e.Graphics.FillEllipse(_foodBrush, _foodSegment.ToRectangle());

            //Draw each segment of the snakes body
            Rectangle[] bodySegments = _snake.ToRectangles;

            foreach (Rectangle bodySegment in bodySegments)
                e.Graphics.FillEllipse(_bodyBrush, bodySegment);

            //Draw the snakes head
            e.Graphics.FillEllipse(_headBrush, _snake.GetHeadSegment().ToRectangle());

            if (IsHitFood())
            {
                _snake.Add();
                _score += 1;
                _isFoodSegmentVisible = false;
                _timer.Stop();
                _timer.Interval = 2000;
                _timer.Start();
            }

            //Draw Score
            e.Graphics.DrawString(string.Format("Score: {0}", _score), new Font(FontFamily.GenericSansSerif, 12, FontStyle.Regular), _scoreBrush, _canvas.Width - 95, 5);

            Thread.Sleep(100);

            //_snake.Add();

            Application.DoEvents();

            _canvas.Invalidate();
        }

        private void MoveSnake()
        {
            _snake.Move();
        }

        private bool IsGameOver()
        {
            Segment headSegment = _snake.GetHeadSegment();

            if (headSegment.X < 0 && _snake.MovingDirection == Direction.Left) //Left side
            {
                return true;
            }
            if (headSegment.Y < 0 && _snake.MovingDirection == Direction.Up) //Top side
            {
                return true;
            }
            if (headSegment.X > _canvas.Width - _snake.Width && _snake.MovingDirection == Direction.Right) //Right side
            {
                return true;
            }
            if (headSegment.Y > _canvas.Height - _snake.Height && _snake.MovingDirection == Direction.Down) //Bottom side
            {
                return true;
            }
            if (IsHitSelf())
            {
                return true;
            }

            return false;
        }

        private bool IsHitFood()
        {
            if (!_isFoodSegmentVisible) return false;

            Segment headSegment = _snake.GetHeadSegment();
            return Segment.Equals(headSegment, _foodSegment);
        }

        private bool IsHitSelf()
        {
            Segment headPosition = _snake.GetHeadSegment();
            return _snake.GetSnakeBodyAsSegments.Any(segment => Segment.Equals(segment, headPosition));
        }

        private void InitializeGrid()
        {
            _grid = new List<Segment>();
            for (int i = 0; i < _canvas.Width; i += 10)
            {
                for (int i2 = 0; i2 < _canvas.Height; i2 += 10)
                {
                    var gridSegment = new Segment(10, 10, i, i2);
                    _grid.Add(gridSegment);
                }
            }
        }

        private Segment GetRandomSegment()
        {
            List<Segment> segments = _snake.ToSegments;

            Segment randomSegment = null;
            bool foundSuitableSegment = false;

            while (!foundSuitableSegment)
            {
                randomSegment = _grid[_random.Next(0, _grid.Count)];
                foreach (Segment s in segments)
                {
                    if (!Segment.Equals(s, randomSegment))
                    {
                        foundSuitableSegment = true;
                    }
                }
            }

            return randomSegment;
        }

        private bool IsCompatible(Direction newDirection)
        {
            if (_currentDirection == newDirection)
                return false;

            if (_currentDirection == Direction.Left && newDirection == Direction.Right)
                return false;

            if (_currentDirection == Direction.Right && newDirection == Direction.Left)
                return false;

            if (_currentDirection == Direction.Up && newDirection == Direction.Down)
                return false;

            if (_currentDirection == Direction.Down && newDirection == Direction.Up)
                return false;

            return true;
        }

        private void Timer_Tick(object sender, EventArgs e)
        {
            if (!_isFoodSegmentVisible)
            {
                _timer.Interval = 10000;
                _foodSegment = GetRandomSegment();
                _isFoodSegmentVisible = true;
            }
            else
            {
                _timer.Interval = 1000;
                _isFoodSegmentVisible = false;
            }
        }
    }

    public enum Direction
    {
        Left,
        Right,
        Up,
        Down
    }
}